globals [
    objective-q
    best-tour-length
    best-tour
    current-best-tour-length
    current-best-tour
    best-ant
    average-tour-length
    initial-pheromone-level
    point-color
    point-label-color
    point-size
    ant-color
    ant-size
    current-best-tour-color
    link-color
    best-tour-color
    average-tour-color
]

breed [ants ant]

breed [points point]

ants-own [
    starting-point
    visited-points
    transited-links
    tour-length
    current-point
]

links-own [
    pheromone-level
    weight
]

to setup-globals
    set point-color yellow
    set point-label-color blue
    set point-size 5
    set ant-color red
    set ant-size 8
    set best-tour-color (green + 2)
    set current-best-tour-color (green - 3)
    set link-color (gray - 4)
    set average-tour-color blue
    set initial-pheromone-level 0.01
    set-default-shape ants "bug"
    set-default-shape points "circle"
end

to startup
    clear-all
    setup-globals
end

to setup-random-points
    no-display
    startup
    create-points number-points [
        initialize-point random-xcor random-ycor
    ]
    reset
    display
end

to import-point-list
    no-display
    let source user-file
    if (not is-boolean? source) [
        file-open source
        if (not file-at-end?) [
            let point-locations file-read
            if (is-list? point-locations) [
                startup
                foreach point-locations [
                    create-points 1 [
                        initialize-point (first ?) (last ?)
                    ]
                ]
                reset
            ]
        ]
        file-close
    ]
    display
end

to import-tsp
    no-display
    let source user-file
    if (not is-boolean? source) [
        let point-locations []
        let in-data-read-mode? false
        let eof-tag "EOF"
        let node-section-tag "NODE_COORD_SECTION"
        file-open source
        startup
        while [not file-at-end?] [
            let input-line file-read-line
            ifelse (input-line = node-section-tag) [
                set in-data-read-mode? true
            ] [
                ifelse (input-line = eof-tag) [
                    set in-data-read-mode? false
                ] [
                    if (in-data-read-mode?) [
                        let input-list (read-from-string (word "[" input-line "]"))
                        if (is-number? first input-list) [
                            set input-list (butfirst input-list)
                            set point-locations (fput input-list point-locations)
                        ]
                    ]
                ]
            ]
        ]
        file-close
        if (is-list? point-locations) [
            let min-x (min map [first ?] point-locations)
            let max-x (max map [first ?] point-locations)
            let min-y (min map [last ?] point-locations)
            let max-y (max map [last ?] point-locations)
            if ((min-x < min-pxcor - 0.5) or (max-x >= max-pxcor + 0.5)
                    or (min-y < min-pycor - 0.5) or (max-y >= max-pycor + 0.5)) [
                let data-x-range (max-x - min-x)
                let data-y-range (max-y - min-y)
                show (list (list min-x min-y) (list max-x max-y))
                let scaling-factor (min list ((world-width - 1) / data-x-range) 
                    ((world-height - 1) / data-y-range))
                let x-offset ((world-width - 1 + data-x-range * scaling-factor) / 2)
                let y-offset ((world-height - 1 + data-y-range * scaling-factor) / 2)
                set point-locations (map [list ((first ? - min-x) * scaling-factor + x-offset)
                    ((last ? - min-y) * scaling-factor + y-offset)] point-locations)
            ]
            set point-locations (reverse point-locations)
            startup            
            foreach point-locations [
                create-points 1 [
                    initialize-point (first ?) (last ?)
                ]
            ]
            reset
        ]
    ]
    display
end

to export-point-list
    if ((count points) = 0) [
        user-message "There are no points to export."
        stop
    ]
    let target user-new-file
    if (not is-boolean? target) [
        if (file-exists? target) [
            file-delete target
        ]
        file-open target
        file-print "["
        foreach (sort points) [
            file-print (word "\t[" ([xcor] of ?) "\t" ([ycor] of ?) "]")
        ]
        file-print "]"
        file-close
    ]
end

to export-tsp
    if ((count points) = 0) [
        user-message "There are no points to export."
        stop
    ]
    let tsp-title (user-input "Provide a name for the TSP data set")    
    if (not is-boolean? tsp-title) [
        let target user-new-file
        if (not is-boolean? target) [
            if (file-exists? target) [
                file-delete target
            ]
            let header 
                (list
                    (word "NAME: " tsp-title)
                    "TYPE: TSP"
                    "COMMENT: Exported from TSP Ants NetLogo model"
                    (word "DIMENSION: " (count points))
                    "EDGE_WEIGHT_TYPE: EUC_2D"
                    "NODE_COORD_SECTION"
                )
            file-open target
            file-print (reduce [(word ?1 "\n" ?2)] header)
            foreach (sort points) [
                file-print (word (1 + [who] of ?) " " ([xcor] of ?) " " ([ycor] of ?))
            ]
            file-print "EOF"
            file-close
        ]
    ]
end

to setup-plot
    set-current-plot-pen "Average"
    set-plot-pen-color average-tour-color
    set-current-plot-pen "Current Best"
    set-plot-pen-color current-best-tour-color
    set-current-plot-pen "Best"
    set-plot-pen-color best-tour-color
end

to initialize-point [x y]
    set color point-color
    set size point-size
    setxy x y
    set plabel-color point-label-color
    set plabel (word (who + 1) "  ")
    create-links-with (other points) [
        hide-link
        set color link-color
    ]
end

to initialize-ant
    set hidden? (not view-ant-motion?)
    set color ant-color
    set size ant-size
end

to reset
    reset-ticks
    clear-all-plots
    ask ants [
        die
    ]    
    ask links [
        hide-link
    ]
end

to setup-for-solution
    ask links [
        hide-link
        set color link-color
        set pheromone-level initial-pheromone-level
    ]
    ask points [
        hatch-ants ants-per-point [
            initialize-ant
        ]
    ]
    set best-tour-length 1000000
    set current-best-tour-length best-tour-length
    set best-ant nobody
    set best-tour nobody
    setup-plot
end

to iterate
    if ((count points) = 0) [
        user-message "There are no points defined. Use the \"Import Points\" or \"Random Points\" button to define TSP points."
        stop
    ]
    if ((count ants) = 0) [
        user-message "There are no ants to solve the TSP problem. Use the \"Setup/Reset Ants\" button to create ants."
        stop
    ]
    let tour-lengths []
    let current-best-ant nobody
    update-link-weights
    ifelse (view-ant-motion?) [
        display
        ask-concurrent (ants with [(not use-elitist-strategy?) or (self != best-ant)]) [
            show-turtle
            tour
        ]
    ] [
        no-display
        ask (ants with [(not use-elitist-strategy?) or (self != best-ant)]) [
            hide-turtle
            tour
        ]
    ]
    set tour-lengths ([tour-length] of ants)
    set current-best-ant (min-one-of ants [tour-length])
    set current-best-tour ([transited-links] of current-best-ant)
    set current-best-tour-length ([tour-length] of current-best-ant)
    set average-tour-length ifelse-value (best-ant = nobody) [
        mean tour-lengths
    ] [
        (best-tour-length + sum tour-lengths) / (count ants)
    ]
    if (current-best-tour-length < best-tour-length) [
        set best-tour-length current-best-tour-length
        set best-tour current-best-tour
        if (use-elitist-strategy?) [
            set best-ant current-best-ant
        ]
    ]
    update-pheromone-trails
    update-link-display
    update-plot
    tick
    if (not view-ant-motion?) [
        display
    ]
end

to update-pheromone-trails
    ask links [
        evaporate-pheromone-trails
    ]
    ask ants [
        leave-pheromones
    ]
end

to update-link-display
    ask links [
        ifelse (member? self best-tour) [
            set color best-tour-color
            show-link
        ] [
            ifelse (member? self current-best-tour) [
                set color current-best-tour-color
                show-link
            ] [
                hide-link
                set color link-color
            ]
        ]
    ]
end

to update-plot
    set-current-plot-pen "Average"
    plot average-tour-length
    set-current-plot-pen "Current Best"
    plot current-best-tour-length
    set-current-plot-pen "Best"
    plot best-tour-length
end

to tour
    start-tour
    while [(empty? transited-links) or (current-point != starting-point)] [
        advance-to-next-point
    ]
    set visited-points (reverse visited-points)
    set transited-links (reverse transited-links)
end

to leave-pheromones
    let amount ((10 ^ pheromone-q) / tour-length)
    let index 0
    foreach transited-links [
        ask ? [
            set pheromone-level (pheromone-level + amount)
        ]
    ]
end

to start-tour
    set tour-length 0
    set starting-point (one-of points-here)
    set current-point starting-point
    set visited-points (list current-point)
    set transited-links []
end

to advance-to-next-point
    let total-weight 0
    let weights []
    let probabilities []
    let chosen-point starting-point
    let chosen-link nobody
    let other-points ([self] of (([link-neighbors] of current-point)) 
        with [not member? self [visited-points] of myself])
    if (not empty? other-points) [
        let inverse-sample 0
        let cumulative 0
        foreach other-points [
            let test-link [link-with ?] of current-point
            set weights (fput ([weight] of test-link) weights)
        ]
        set total-weight (sum weights)
        set weights (reverse weights)
        set weights (map [? / total-weight] weights)
        set inverse-sample (random-float 1.0)
        set cumulative 0.0
        while [cumulative <= inverse-sample] [
            set cumulative (cumulative + first weights)
            set weights (butfirst weights)
            set chosen-point (first other-points)
            set other-points (butfirst other-points)
        ]
    ]
    set chosen-link ([link-with chosen-point] of current-point)
    set visited-points (fput chosen-point visited-points)
    set transited-links (fput chosen-link transited-links)
    set tour-length (tour-length + [link-length] of chosen-link)
    face chosen-point
    forward distance chosen-point
    set current-point chosen-point
end

to evaporate-pheromone-trails
    ask links [
        set pheromone-level (max (list initial-pheromone-level ((1 - (evaporation-rate / 100)) * pheromone-level)))
    ]
end

to update-link-weights 
    ask links [
        set weight ((pheromone-level ^ pheromone-weight) / (link-length ^ distance-weight))
    ]
end
@#$#@#$#@
GRAPHICS-WINDOW
300
10
810
541
-1
-1
1.0
1
10
1
1
1
0
0
0
1
0
499
0
499
0
0
1
iterations

SLIDER
35
75
145
108
number-points
number-points
5
50
25
5
1
NIL
HORIZONTAL

SLIDER
820
10
980
43
ants-per-point
ants-per-point
1
50
25
1
1
NIL
HORIZONTAL

BUTTON
155
75
265
108
Random Points
setup-random-points
NIL
1
T
OBSERVER
NIL
P
NIL
NIL

BUTTON
845
340
955
373
Go!
iterate
T
1
T
OBSERVER
NIL
G
NIL
NIL

PLOT
10
288
290
488
TSP Tour Length
Iteration
Length
0.0
10.0
0.0
10.0
true
false
PENS
"Average" 1.0 0 -13345367 true
"Best" 1.0 0 -12087248 true
"Current Best" 1.0 0 -6565750 true

MONITOR
105
495
190
540
Last Min Tour
current-best-tour-length
1
1
11

MONITOR
10
495
95
540
Best Tour
best-tour-length
1
1
11

SLIDER
820
115
980
148
evaporation-rate
evaporation-rate
0.0
100.0
55
0.5
1
%
HORIZONTAL

SLIDER
820
160
980
193
pheromone-weight
pheromone-weight
0
10
4
0.5
1
NIL
HORIZONTAL

SLIDER
820
205
980
238
distance-weight
distance-weight
0
10
1
0.5
1
NIL
HORIZONTAL

BUTTON
35
10
145
43
Import Points
import-point-list
NIL
1
T
OBSERVER
NIL
I
NIL
NIL

SLIDER
820
250
980
283
pheromone-q
pheromone-q
0
5
4
1
1
NIL
HORIZONTAL

SWITCH
820
295
980
328
use-elitist-strategy?
use-elitist-strategy?
0
1
-1000

SWITCH
820
505
980
538
view-ant-motion?
view-ant-motion?
0
1
-1000

BUTTON
35
140
145
173
Export Points
export-point-list
NIL
1
T
OBSERVER
NIL
E
NIL
NIL

BUTTON
845
55
955
88
Setup/Reset Ants
reset\nsetup-for-solution
NIL
1
T
OBSERVER
NIL
S
NIL
NIL

MONITOR
200
495
290
540
Last Avg Tour
average-tour-length
1
1
11

BUTTON
155
140
265
173
Export TSP Format
export-tsp
NIL
1
T
OBSERVER
NIL
T
NIL
NIL

BUTTON
155
10
265
43
Import TSP Format
import-tsp
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This is a model which demonstrates the use of an "ant-based" approach to obtain a reasonably good solution to the Traveling Salesman Problem.

HOW IT WORKS
------------
This section could explain what rules the agents use to create the overall behavior of the model.


HOW TO USE IT
-------------
This section could explain how to use the model, including a description of each of the items in the interface tab.


THINGS TO NOTICE
----------------
This section could give some ideas of things for the user to notice while running the model.


THINGS TO TRY
-------------
This section could give some ideas of things for the user to try to do (move sliders, switches, etc.) with the model.


EXTENDING THE MODEL
-------------------
This section could give some ideas of things to add or change in the procedures tab to make the model more complicated, detailed, accurate, etc.


NETLOGO FEATURES
----------------
This section could point out any especially interesting or unusual features of NetLogo that the model makes use of, particularly in the Procedures tab.  It might also point out places where workarounds were needed because of missing features.


RELATED MODELS
--------------
This section could give the names of models in the NetLogo Models Library or elsewhere which are of related interest.


CREDITS AND REFERENCES
----------------------
This section could contain a reference to the model's URL on the web if it has one, as well as any other necessary credits or references.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1RC5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
